<?php
/*
Plugin Name: Author Stats Pro
plugin URI: https://draftpress.com/?post_type=download&p=8383
Description: Displays post, comment and view stats about all authors on your blog.
version: 2.2.2
Author: DraftPress
Author URI: https://draftpress.com
License: GPL2
*/

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

// Item Name

if (!defined('NNROBOTS_AUTHOR_STATS_ITEM_NAME'))
    define('NNROBOTS_AUTHOR_STATS_ITEM_NAME', 'Author Stats Pro');

// Store URL

if (!defined('NNROBOTS_AUTHOR_STATS_STORE_URL'))
    define('NNROBOTS_AUTHOR_STATS_STORE_URL', 'https://draftpress.com');

// Minimum PHP version

if (!defined('NNROBOTS_AUTHOR_STATS_MIN_PHP_VERSION'))
    define('NNROBOTS_AUTHOR_STATS_MIN_PHP_VERSION', '5.4.0');

// Plugin Name

if (!defined('NNROBOTS_AUTHOR_STATS_PLUGIN_NAME'))
    define('NNROBOTS_AUTHOR_STATS_PLUGIN_NAME', trim(dirname(plugin_basename(__FILE__)), '/'));

// Plugin directory

if (!defined('NNROBOTS_AUTHOR_STATS_PLUGIN_DIR'))
    define('NNROBOTS_AUTHOR_STATS_PLUGIN_DIR', plugin_dir_path(__FILE__) );

// Plugin url

if (!defined('NNROBOTS_AUTHOR_STATS_PLUGIN_URL'))
    define('NNROBOTS_AUTHOR_STATS_PLUGIN_URL', plugins_url() . '/' . NNROBOTS_AUTHOR_STATS_PLUGIN_NAME);

// Plugin verison

if (!defined('NNROBOTS_AUTHOR_STATS_VERSION_NUM'))
    define('NNROBOTS_AUTHOR_STATS_VERSION_NUM', '2.2.2');

// Date Format

if (!defined('NNROBOTS_AUTHOR_STATS_DATE_FORMAT'))
    define('NNROBOTS_AUTHOR_STATS_DATE_FORMAT', 'Y-m-d');

// Activatation / Deactivation

register_activation_hook( __FILE__, 							array('NNRobots_Author_Stats', 'author_stats_activation'));

// Hooks / Filter

add_action('admin_menu', 										array('NNRobots_Author_Stats', 'menu'));
add_action('show_user_profile', 								array('NNRobots_Author_Stats', 'add_field'));
add_action('init', 												array('NNRobots_Author_Stats', 'load_textdoamin'));
add_action('admin_notices', 									array('NNRobots_Author_Stats', 'admin_notices'));
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 	array('NNRobots_Author_Stats', 'settings_link'));

add_action('init', 												array('NNRobots_Author_Stats', 'export_to_csv'));

add_action('save_post', 										array('NNRobots_Author_Stats', 'update_word_count'), 10, 3);
add_action('transition_post_status', 							array('NNRobots_Author_Stats', 'update_post_status'), 10, 3 );

// AJAX functions

add_action('wp_ajax_nnr_as_stats_load', 						array('NNRobots_Author_Stats', 'stats_load'));
add_action('wp_ajax_nopriv_nnr_as_stats_load', 					array('NNRobots_Author_Stats', 'stats_load'));
add_action('wp_ajax_nnr_as_user_profile_stats', 				array('NNRobots_Author_Stats', 'get_user_profile_stats'));

// Include files

require('admin/model/db.php');
require('widget.php');
require('shortcode.php');

// Shortcode

add_shortcode('nnr_author_leaderboard', 'nnr_as_leaderboard_shortcode');

// Software Licensing Updates

add_action('wp_ajax_nnr_as_license_action', 	array('NNRobots_Author_Stats', 'license_action'));

if ( !class_exists('EDD_SL_Plugin_Updater') ) {
	include_once( dirname( __FILE__ ) . '/admin/include/edd-software-licensing-updates/EDD_SL_Plugin_Updater.php' );
}

$edd_updater = new EDD_SL_Plugin_Updater( NNROBOTS_AUTHOR_STATS_STORE_URL , __FILE__, array(
		'version' 	=> NNROBOTS_AUTHOR_STATS_VERSION_NUM,
		'license' 	=> NNRobots_Author_Stats::get_license_key(),
		'item_name' => NNROBOTS_AUTHOR_STATS_ITEM_NAME,
		'author' 	=> '99 Robots',
	)
);

/**
 * Index all the posts on the site
 *
 * @access public
 * @return void
 */
function nnr_as_index_all_posts_callback($page) {
	include_once( NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . 'admin/model/indexed-posts.php');
	NNR_AS_Indexed_Post::index_posts($page);
}
add_action( 'nnr_as_index_all_posts', 'nnr_as_index_all_posts_callback' );

/**
 *
 * Main class for the Gabfire Author Stats Widget
 *
 * @since 1.0.0
 */
class NNRobots_Author_Stats {

	/**
	 * prefix
	 *
	 * (default value: 'nnrobots_author_stats_')
	 *
	 * @var string
	 * @access public
	 * @static
	 */
	static $prefix = 'nnrobots_author_stats_';

	/**
	 * prefix_dash
	 *
	 * (default value: 'nnr-as-')
	 *
	 * @var string
	 * @access public
	 * @static
	 */
	static $prefix_dash = 'nnr-as-';

	/**
	 * text_domain
	 *
	 * (default value: '99robots-author-stats')
	 *
	 * @var string
	 * @access public
	 * @static
	 */
	static $text_domain = '99robots-author-stats';

    /**
     * settings_page
     *
     * (default value: '99robots-author-stats-settings')
     *
     * @var string
     * @access public
     * @static
     */
    static $settings_page = '99robots-author-stats-settings';

	/**
     * default
     *
     * Default values for widget
     *
     * @var string
     * @access public
     * @static
     */
    static $default = array(
		'title'			=> 'Author Stats',
		'post_types'	=> 'post,page,attachment',
		'user_roles'	=> 'Administrator,Editor,Author,Contributor',
		'post'			=> true,
		'view'			=> true,
		'comment'		=> true,
		'rank'			=> 'views',
		'top'			=> 5,
		'date_range'	=> 'all_time',
		'avatar_shape'	=> 'rounded-square',
		'avatar_size'	=> 'large',
	);

    /**
     * Return the default option
     *
     * @access public
     * @static
     * @return void
     */
    static function default_options() {

	    global $wp_roles;

	    $settings = array();

	    foreach ( $wp_roles->roles as $wp_role ) {
			$role = str_replace(' ', '', strtolower($wp_role['name']));
			$settings['user_role_' .  $role] = true;
		}

		return $settings;
    }

	/**
	 * Hooks into 'register_activation_hook' to create db table when plugin is activated
	 *
	 * @since 1.0.0
	 *
	 * @param 	N/A
	 * @return 	N/A
	 */
	static function author_stats_activation() {

		// Check if multisite, if so then save as site option

		if (is_multisite()) {
			update_site_option(self::$prefix . 'version', NNROBOTS_AUTHOR_STATS_VERSION_NUM);
		} else {
			update_option(self::$prefix . 'version', NNROBOTS_AUTHOR_STATS_VERSION_NUM);
		}

		update_option(self::$prefix . 'activation_notice', true);

		// Create the tables

		include_once( NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . 'admin/model/db.php');
		include_once( NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . 'admin/model/indexed-posts.php');

		NNRobots_Author_Stats_Table::createTable();
		NNR_AS_Indexed_Post::createTable();

		// Start the indexing of all posts

		wp_schedule_single_event( time() , 'nnr_as_index_all_posts', array(1) );
		set_transient('nnr_as_indexed_posts', true);
	}

	/**
	 * Hooks to 'plugin_action_links_' filter
	 *
	 * @since 1.0.0
	 */
	static function settings_link($links) {
		$settings_link = '<a href="users.php?page=' . self::$settings_page . '">Stats</a>';
		array_unshift($links, $settings_link);
		return $links;
	}

	/**
	 * Load the text domain
	 *
	 * @since 1.0.0
	 */
	static function load_textdoamin() {
		load_plugin_textdomain(self::$text_domain, false, basename( dirname( __FILE__ ) ) . '/languages' );
	}

	/**
	 * Hook to create an admin notice
	 *
	 * @since 1.0.0
	 *
	 * @param 	N/A
	 * @return 	N/A
	 */
	static function admin_notices() {

		// Please re-activate plugin

		global $wpdb;

		$result = $wpdb->get_results("SHOW TABLES LIKE '" . $wpdb->prefix . "nnr_as_indexed_posts'");

		if ( count($result) < 1 ) {
			?> <div class="error">
				<p><?php _e('Author Stats Pro needs to be de-activated and then re-activated in order for it to perform necessary functions.', self::$text_domain);
				?></p>
			</div> <?php
		}

		// Indexing posts

        $indexed = get_transient('nnr_as_indexed_posts');

		if ( false !== $indexed ) {
			?> <div class="updated nnr-notice-updated">
				<p><?php _e('Author Stats Pro is currently indexing your posts.  We will notify ' . get_option('admin_email') . ' when the process has completed.', self::$text_domain);
				?></p>
			</div> <?php
		}

		// Data calculation notice

		if ( current_user_can('manage_options') ) {

			$activation_notice = get_option(self::$prefix . 'activation_notice');

			if ( $activation_notice !== false ) {
				?> <div class="updated nnr-notice-updated">
					<p><?php _e('Author Stats Pro plugin calculates data only when it is activated. Therefore, upon initial activation, there will be zero data. Deactivating this plugin will NOT delete existing data. | <a href="users.php?page=' . self::$settings_page . '&' . self::$prefix . 'notice=0' . '">Hide Notice</a>', self::$text_domain);
					?></p>
				</div> <?php
			}

            $indexed_notice = get_option(self::$prefix . 'indexed_notice');

            if ( !$indexed_notice && !$indexed ) {
                ?> <div class="updated nnr-notice-updated">
                    <p><?php _e('Author Stats Pro has successfully completed indexing and generating stats. | <a href="users.php?page=' . self::$settings_page . '&' . self::$prefix . 'indexed_notice=0' . '">Hide Notice</a>', self::$text_domain);
                        ?></p>
                </div> <?php
            }

		}

        // Show error message is WP Cron is turned off

        if ( defined('DISABLE_WP_CRON') && DISABLE_WP_CRON ) {
            ?>
            <div class="error nnr-notice-updated">
                <p><?php _e('<strong>WP Cron</strong> is <strong>DISABLED</strong>! <strong>Author Stats Pro</strong> needs WP Cron to be enabled in order to automatically index your content. Please read our' , self::$text_domain); ?> <a href="https://99robots.com/?post_type=doc&p=9799" target="_blank"><?php _e('post' , self::$text_domain); ?></a> <?php _e('about how to enable WP Cron.' , self::$text_domain); ?></p>
            </div>
            <?php
        }
	}

	/**
	 * Perform all the actions with a license
	 *
	 * @access public
	 * @static
	 * @return void
	 */
	static function license_action() {

		if ( !isset($_POST['license_key']) ) {
			echo '';
			die();
		}

		// data to send in our API request

		$api_params = array(
			'edd_action'	=> $_POST['license_action'],
			'license' 		=> $_POST['license_key'],
			'item_name' 	=> urlencode( NNROBOTS_AUTHOR_STATS_ITEM_NAME ), // the name of our product in EDD
			'url'       	=> is_multisite() ? network_home_url() : home_url()
		);

		// Call the custom API.

		$response = wp_remote_post( NNROBOTS_AUTHOR_STATS_STORE_URL, array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );

		// make sure the response came back okay

		if ( is_wp_error( $response ) ) {
			echo $response;
			die();
		}

		$license_data = json_decode( wp_remote_retrieve_body( $response ) );

		if ( $_POST['license_action'] == 'activate_license' ) {
			self::update_license_key($_POST['license_key']);
		} else if ( $_POST['license_action'] == 'deactivate_license' && $license_data->license == 'deactivated') {
			self::update_license_key('');
		}

		// decode the license data

		echo $license_data->license;

		die();

	}

	/**
	 * Check if license key is active
	 *
	 * @access public
	 * @static
	 * @param mixed $license_key
	 * @return void
	 */
	static function is_license_active( $license_key ) {

		if ( !isset( $license_key ) ) {
			return false;
		}

		// data to send in our API request

		$api_params = array(
			'edd_action'	=> 'check_license',
			'license' 		=> $license_key,
			'item_name' 	=> urlencode( NNROBOTS_AUTHOR_STATS_ITEM_NAME ), // the name of our product in EDD
			'url'       	=> is_multisite() ? network_home_url() : home_url()
		);

		// Call the custom API.

		$response = wp_remote_post( NNROBOTS_AUTHOR_STATS_STORE_URL, array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );

		// make sure the response came back okay

		if ( is_wp_error( $response ) ) {
			return false;
		}

		$license_data = json_decode( wp_remote_retrieve_body( $response ) );

		// decode the license data

		if ( isset($license_data->license) && $license_data->license == 'valid' ) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * Get the license key
	 *
	 * @access public
	 * @static
	 * @return void
	 */
	static function get_license_key() {

		// Get the license key from based on WordPress install

		if ( function_exists('is_multisite') && is_multisite() ) {
			$license_key = get_site_option(self::$prefix . 'license_key');
		} else {
			$license_key = get_option(self::$prefix . 'license_key');
		}

		// Check if license key is set

		if ( $license_key === false ) {
			$license_key = '';
		}

		return $license_key;
	}

	/**
	 * Update the license key
	 *
	 * @access public
	 * @static
	 * @param mixed $license_key
	 * @return void
	 */
	static function update_license_key($license_key) {

		if ( isset( $license_key ) ) {

			// Get the license key from based on WordPress install

			if ( function_exists('is_multisite') && is_multisite() ) {

				update_site_option(self::$prefix . 'license_key', $license_key);

			} else {

				update_option(self::$prefix . 'license_key', $license_key);

			}
		}
	}

	/**
	 * Hook for the admin menu
	 *
	 * @since 1.0.0
	 *
	 * @param 	N/A
	 * @return 	N/A
	 */
	static function menu() {

		// Settings page

		$settings_page_load = add_submenu_page(
	 		'users.php',
	 		__("Author Stats Pro", self::$text_domain),
	 		__("Author Stats Pro", self::$text_domain),
	 		'manage_options',
	 		self::$settings_page,
	 		array('NNRobots_Author_Stats','stats'));

	 	add_action("admin_print_scripts-$settings_page_load", array('NNRobots_Author_Stats', 'admin_include_scripts'));
	}

	/**
	 * include_scripts function.
	 *
	 * Include css on admin
	 * @access public
	 * @static
	 * @return void
	 */
	static function admin_include_scripts() {

		// Styles
		wp_enqueue_style(self::$prefix . 'settings_css', 					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/css/settings.css');
		wp_enqueue_style(self::$prefix . 'bootstrap_css', 					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/css/nnr-bootstrap.min.css');
		wp_enqueue_style(self::$prefix . 'fontawesome_css',					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/include/fontawesome/css/font-awesome.min.css');
		wp_enqueue_style(self::$prefix . 'bootstrap_sortable_css', 			NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/css/bootstrap-sortable.css');
		wp_enqueue_style(self::$prefix . 'bootstrap_selectize_css', 		NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/css/selectize.bootstrap3.css');
        wp_enqueue_style( 'jquery_ui_css', NNROBOTS_AUTHOR_STATS_PLUGIN_URL. '/admin/css/jquery-ui.css' );

        // Scripts
        wp_enqueue_script( 'jquery-ui-datepicker' );
		wp_enqueue_script(self::$prefix . 'bootstrap_js', 					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/bootstrap.min.js', array('jquery'));
		wp_enqueue_script(self::$prefix . 'bootstrap_sortable_js', 			NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/bootstrap-sortable.js', array('jquery', self::$prefix . 'bootstrap_moment_js'));
		wp_enqueue_script(self::$prefix . 'bootstrap_moment_js', 			NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/moment.js', array('jquery'));
		wp_enqueue_script(self::$prefix . 'bootstrap_selectize_js', 		NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/selectize.min.js', array('jquery'));
		wp_enqueue_script(self::$prefix . 'settings_js', 					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/settings.js', array('jquery'));
		wp_localize_script(self::$prefix . 'settings_js', 					'nnr_as_data', array(
			'prefix_dash'			=> self::$prefix_dash,
			'activate_text'			=> __('Activate', self::$text_domain),
			'deactivate_text'		=> __('Deactivate', self::$text_domain),
			'no_license_key'		=> __('Please add or activate your License Key to get automatic updates.  Without a valid license key you will not receive regular updates. You can find your license key', self::$text_domain) . ' <a href="https://99robots.com/dashboard" target="_blank">here</a>',
			'expired'				=> __('EXPIRED: This license key has expired.  Please renew your license key', self::$text_domain) . ' <a href="https://99robots.com/dashboard" target="_blank">here</a>',
			'activation_error'		=> __('INVALID LICENSE KEY: The license key is not valid please try again. You can find your license key', self::$text_domain) . ' <a href="https://99robots.com/dashboard" target="_blank">here</a>.',
			'activation_valid'		=> __('SUCCESS: Your license key is valid.', self::$text_domain),
			'deactivation_valid'	=> __('SUCCESS: This site has been deactivated.', self::$text_domain),
			'deactivation_error'	=> __('DEACTIVATION FAILED: This site could not be deactivated.', self::$text_domain),
		));

		// User Roles

		global $wp_roles;
		$roles = array();

		foreach ( $wp_roles->roles as $role ) {

			$role['name'] = str_replace('|User role', '', $role['name']);

			if ( isset($role['capabilities']['edit_posts']) && $role['capabilities']['edit_posts'] ) {
				$roles[] = $role['name'];
			}

		}

		wp_enqueue_script(self::$prefix . 'stats_js', 					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/stats.js?v=0.1', array('jquery'));
		wp_localize_script(self::$prefix . 'stats_js', 					'nnr_as_stats_data', array(
			'prefix_dash'			=> self::$prefix_dash,
			'today'					=> date('m/d/Y'),
			'post_types'			=> get_post_types(array('public' => true), 'names'),
			'user_roles'			=> $roles,
		));
	}

	/**
	 * Displays on the dashboard
	 *
	 * @since 1.0.0
	 *
	 * @param 	N/A
	 * @return 	N/A
	 */
	static function stats() {

		// Notice

		if ( isset($_GET['nnrobots_author_stats_notice']) && $_GET['nnrobots_author_stats_notice'] == '0' ) {
			delete_option(self::$prefix . 'activation_notice');
			?>
			<script type="text/javascript">
			jQuery(document).ready(function($){
				window.location = "<?php echo get_admin_url(); ?>users.php?page=<?php echo self::$settings_page; ?>";
			});
			</script>
			<?php
		}

        if ( isset($_GET['nnrobots_author_stats_indexed_notice']) && $_GET['nnrobots_author_stats_indexed_notice'] == '0' ) {
            update_option(self::$prefix . 'indexed_notice', 'yes');
            ?>
            <script type="text/javascript">
                jQuery(document).ready(function($){
                    window.location = "<?php echo get_admin_url(); ?>users.php?page=<?php echo self::$settings_page; ?>";
                });
            </script>
            <?php
        }

		global $wp_version;
		global $wp_roles;

		$license_key = self::get_license_key();

		$settings = get_option(self::$prefix . 'settings');

		// Get settings

		if ( $settings == false ) {
			$settings = self::default_options();
		}

		// Format dates

		$start_date = isset($_GET[self::$prefix_dash . "start-date"]) && $_GET[self::$prefix_dash . "start-date"] != '' ?
						date('m/d/Y', strtotime($_GET[self::$prefix_dash . "start-date"])) :
						date('m/d/Y', mktime(0, 0, 0, date("m"), date("d"), date("Y")-1));

		$end_date = isset($_GET[self::$prefix_dash . "end-date"]) && $_GET[self::$prefix_dash . "end-date"] != '' ?
						date('m/d/Y', strtotime($_GET[self::$prefix_dash . "end-date"])) :
						date("m/d/Y", strtotime(current_time('mysql')));

		// Save Settings

		if ( isset($_POST['submit-settings']) && check_admin_referer(self::$prefix . 'admin_settings') ) {

			foreach ( $wp_roles->roles as $wp_role ) {

				$role = str_replace(' ', '', strtolower($wp_role['name']));

				$settings['user_role_' .  $role] = isset($_POST[self::$prefix_dash . 'user-role-' . $role]) && $_POST[self::$prefix_dash . 'user-role-' . $role] ? true : false;
			}

			update_option(self::$prefix . 'settings', $settings);

		}

		// Post Types

		$post_types = array('post');// get_post_types(array('public' => true), 'names');

		// User Roles

		global $wp_roles;
		$roles = array();

		if ( isset($_POST['user_roles']) && $_POST['user_roles'] != '' ) {
			$roles = explode(',', urldecode($_POST['user_roles']));
		} else {
			foreach ( $wp_roles->roles as $role ) {

				$role['name'] = str_replace('|User role', '', $role['name']);

				if ($role['name'] == 'Administrator' ||
					$role['name'] == 'Editor' ||
					$role['name'] == 'Author' ||
					$role['name'] == 'Contributor' ||
					$role['name'] == 'Super Admin' ) {

					$roles[] = $role['name'];
				}
			}
		}

		// Reset Stats

		if ( isset($_GET['action']) && $_GET['action'] == 'reset-stats' ) {

			global $wpdb;
			$wpdb->query('DROP TABLE `' . $wpdb->prefix . 'nnrobots_author_stats`');

			$params = http_build_query(array(
				'error_message'	=> urlencode(__('Stats have been reset.', self::$text_domain)),
				'status'		=> 'danger',
			));

			?>
			<script type="text/javascript">
			jQuery(document).ready(function($){
				window.location = "<?php echo get_admin_url(); ?>users.php?page=<?php echo self::$settings_page . '&' . $params; ?>";
			});
			</script>
			<?php

		}

		require('admin/views/settings.php');

	}

	/**
	 * Load the stats asychronously
	 *
	 * @access public
	 * @static
	 * @return void
	 */
	static function stats_load( $export = false ) {

		// Parse data based on HTTP call
		if ( !$export ) {
            $data_filter_metric = !empty($_POST['filter_metric']) ? urldecode($_POST['filter_metric']) : 'posts';
            $data_start_date = isset($_POST['start_date']) ? urldecode($_POST['start_date']) : null;
			$data_end_date   = isset($_POST['end_date']) ? urldecode($_POST['end_date']) : null;
			$data_post_types = isset($_POST['post_types']) ? urldecode($_POST['post_types']) : null;
			$data_user_roles = isset($_POST['user_roles']) ? urldecode($_POST['user_roles']) : null;
			$data_author     = isset($_POST['author']) ? $_POST['author'] : '';
		} else {
            $data_filter_metric = !empty($_GET[self::$prefix_dash . 'filter-metric']) ? urldecode($_GET[self::$prefix_dash . 'filter-metric']) : 'posts';
            $data_start_date = isset($_GET[self::$prefix_dash . 'start-date']) ? urldecode($_GET[self::$prefix_dash . 'start-date']) : null;
			$data_end_date   = isset($_GET[self::$prefix_dash . 'end-date']) ? urldecode($_GET[self::$prefix_dash . 'end-date']) : null;
			$data_post_types = isset($_GET[self::$prefix_dash . 'post-types']) ? urldecode($_GET[self::$prefix_dash . 'post-types']) : null;
			$data_user_roles = isset($_GET[self::$prefix_dash . 'user-roles']) ? urldecode($_GET[self::$prefix_dash . 'user-roles']) : null;
			$data_author     = isset($_GET['author']) ? $_GET['author'] : '';
		}

		include_once(NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . "/admin/model/db.php");
		include_once(NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . 'admin/model/indexed-posts.php');

		// Get Start and End Dates

		$start_date = date(NNROBOTS_AUTHOR_STATS_DATE_FORMAT, mktime(0, 0, 0, date("m"), date("d"), date("Y")-1));
		$end_date = date(NNROBOTS_AUTHOR_STATS_DATE_FORMAT, strtotime(current_time('mysql')));

		if (isset($data_start_date) && $data_start_date != '') {
			$start_date = date(NNROBOTS_AUTHOR_STATS_DATE_FORMAT, strtotime($data_start_date));
		}

		if (isset($data_end_date) && $data_end_date != '') {
			$end_date = date(NNROBOTS_AUTHOR_STATS_DATE_FORMAT, strtotime($data_end_date));
		}

		// Post Types

		if ( isset($data_post_types) && $data_post_types != '' ) {
			$post_types = explode(',', $data_post_types);
		} else {
			$post_types = array('post');// get_post_types(array('public' => true), 'names');
		}

		// User Roles

		global $wp_roles;
		$roles = array();

		if ( isset($data_user_roles) && $data_user_roles != '' ) {
			$roles = explode(',', $data_user_roles);
		} else {
			foreach ( $wp_roles->roles as $role ) {

				$role['name'] = str_replace('|User role', '', $role['name']);

				if ($role['name'] == 'Administrator' ||
					$role['name'] == 'Editor' ||
					$role['name'] == 'Author' ||
					$role['name'] == 'Contributor' ||
					$role['name'] == 'Super Admin' ) {

					$roles[] = $role['name'];
				}
			}
		}

		$authors = get_users(array('who' => 'contributors'));
		$valid_authors = array();

		foreach ( $authors as $author ) {
			foreach ($roles as $role) {
				if ( in_array( strtolower($role) , $author->roles) ) {
					$valid_authors[] = $author;
					break;
				}
			}
		}

		$authors = $valid_authors;

		if($data_filter_metric == "views") {
            $stats = NNRobots_Author_Stats_Table::get_views($start_date, $end_date);
        } else {
            $stats = NNRobots_Author_Stats_Table::get_views();
        }

		$author_views = array();

		foreach ( $authors as $author ) {

			if ( !isset($author_views[$author->ID]) ) {
				$author_views[$author->ID] = 0;
			}

			foreach ( $stats as $stat ) {

				if ( $stat['author'] == $author->ID && in_array($stat['post_type'], $post_types) ) {
					$author_views[$author->ID] += $stat['views'];
				}
			}
		}

		$settings = get_option(self::$prefix . 'settings');

		// Get settings

		if ( $settings == false ) {
			$settings = self::default_options();
		}

		global $wpdb;

		// All Authors

		if ( !isset($data_author) || ( isset($data_author) && $data_author == '' ) ) {

			$author_stats_data = array();

			$content = '<div class="' . self::$prefix_dash . 'table-container">
			<table class="table table-striped table-responsive sortable">
				<thead>
					<tr>
						<th>' . __('Authors',self::$text_domain) . '</th>
						<th>' . __('User Role',self::$text_domain) . '</th>
						<th>' . __('Posts',self::$text_domain) . '</th>
						<th>' . __('Average Word Count',self::$text_domain) . '</th>
						<th>' . __('Word Count',self::$text_domain) . '</th>
						<th data-defaultsort="desc">' . __('Views',self::$text_domain) . '</th>
						<th>' . __('Comments',self::$text_domain) . '</th>';

					$content .= '</tr>
				</thead>

				<tbody>';

					foreach ($authors as $author) {

						// Avatar

						$avatar = get_avatar($author->ID, 50);

						// Get word count

                        if($data_filter_metric == "posts") {
                            $posts_by_author = NNR_AS_Indexed_Post::get_posts_by_author($author->ID, $post_types, $start_date, $end_date);
                        } else {
                            $posts_by_author = NNR_AS_Indexed_Post::get_posts_by_author($author->ID, $post_types);
                        }

						$word_count = 0;

						foreach ( $posts_by_author as $post ) {
							$word_count += (int) $post['word_count'];
						}

                        if($data_filter_metric == "views") {
                            $comments = get_comments(array(
                                'post_author' 	=> $author->ID,
                                'count' 		=> true,
                                'date_query' 	=> array(
                                    array(
                                        'after'     => $start_date,
                                        'before'    => $end_date,
                                        'inclusive' => true,
                                    ),
                                )
                            ));
                        } else {
                            $comments = get_comments(array(
                                'post_author' 	=> $author->ID,
                                'count' 		=> true
                            ));
                        }

						$word_density = (count($posts_by_author) != 0 ? round($word_count / count($posts_by_author), 2) : 0);

						// Content

						$content .= '<tr>
							<td data-value="' . $author->display_name . '">
								<div class="' . self::$prefix_dash . 'author-avatar">';

									// Display avatar

									$content .= $avatar;

									// Name and ID

									$content .= '<div class="' . self::$prefix_dash . 'author-data">' .
										$author->display_name .
										' <em class="' . self::$prefix_dash . 'author-id">id:(' . $author->ID . ')</em>';

										// Row Actions

										$content .= '<div><!-- row-actions ' . self::$prefix_dash . 'row-actions -->
											<span class="edit">
												<a href="' . get_admin_url() . 'users.php?page=' . self::$settings_page . '&author=' . $author->ID . '&' . self::$prefix_dash . 'filter-metric=' . $data_filter_metric . '&' . self::$prefix_dash . 'start-date=' . $start_date . '&' . self::$prefix_dash . 'end-date=' . $end_date . '&' . self::$prefix_dash . 'post-types=' . urlencode(implode(',', $post_types)) . '&user-roles=' . urlencode(implode(',', $roles)) . '">' . __('View User Stats', self::$text_domain) . '</a>
											</span>
										</div>';

									$content .= '</div>
									</div>' .
							'</td>
							<td data-value="' . $author->roles[0] . '">' . $author->roles[0] . '</td>
							<td data-value="' . count($posts_by_author) . '">' . number_format(count($posts_by_author)) . '</td>
							<td data-value="' . $word_density . '">' . number_format($word_density) . '</td>
							<td data-value="' . $word_count . '">' . number_format($word_count) . '</td>
							<td data-value="' . $author_views[$author->ID] . '">' . number_format($author_views[$author->ID]) . '</td>
							<td data-value="' . $comments . '">' . number_format($comments) . '</td>';

						$content .= '</tr>';

						$author_stats_data_additions = array(
							$author->ID,
							$author->user_login,
							$author->display_name,
							count($posts_by_author),
							number_format($word_density),
							$word_count,
							$author_views[$author->ID],
							$comments,
						);

						$author_stats_data[] = $author_stats_data_additions;

					}

				$content .= '</tbody>
			</table>';

			$content .= '</div>';
		}

		// Single Author

		else {

			$author_stats_data = array();

			$author_id = (int) $data_author;
			$author_object = get_user_by('id', $author_id);

			$posts_by_author = NNR_AS_Indexed_Post::get_posts_by_author($author_id, $post_types, $start_date, $end_date);

			// Avatar

			$avatar = get_avatar($author_id, 50);

			$author_data =
				'<div class="' . self::$prefix_dash . 'author-avatar ' . self::$prefix_dash . 'single-author">' . $avatar . '</div>' .
				'<div class="' . self::$prefix_dash . 'author-data ' . self::$prefix_dash . 'single-author">' .
					'<div>' . $author_object->display_name . '</div>' .
					'<em class="' . self::$prefix_dash . 'author-id">id:(' . $author_id . ')</em>' .
				'</div>';

			$content = '<div class="clearfix"></div>
			<div class="' . self::$prefix_dash . 'table-container">
			<table class="table table-striped sortable">
				<thead>
					<tr>
						<th>' . __('ID',self::$text_domain) . '</th>
						<th>' . __('Published Date',self::$text_domain) . '</th>
						<th>' . __('Post Type',self::$text_domain) . '</th>
						<th>' . __('Post Title',self::$text_domain) . '</th>
						<th>' . __('Word Count',self::$text_domain) . '</th>
						<th data-defaultsort="desc">' . __('Views *',self::$text_domain) . '</th>
						<th>' . __('Comments *',self::$text_domain) . '</th>';

					$content .= '</tr>
				</thead>

				<tbody>';

					$total_word_count = 0;

					foreach ($posts_by_author as $post ) {

						// Get word count

						$total_word_count += $post['word_count'];

					    $comments = get_comments(array(
							'post_id' 		=> $post['post'],
							'count' 		=> true,
							'date_query' 	=> array(
									array(
										'after'     => $start_date,
										'before'    => $end_date,
										'inclusive' => true,
								),
						)));

						$post_views = NNRobots_Author_Stats_Table::get_views_by_post($post['post'], $start_date, $end_date);

						// Content

						$content .= '<tr>
							<td data-value="' . $post['post'] . '">' . $post['post'] . '</td>
							<td data-value="' . strtotime($post['published_date']) . '">' . date('F j, Y', strtotime($post['published_date'])) . '</td>
							<td data-value="' . $post['post_type'] . '">' . $post['post_type'] . '</td>
							<td data-value="' . $post['title'] . '"><a href="' . get_the_permalink($post['post']) . '" target="_blank">' . wp_trim_words($post['title'], 10, '...') . '</a></td>
							<td data-value="' . $post['word_count'] . '">' . number_format($post['word_count']) . '</td>
							<td data-value="' . $post_views . '">' . number_format($post_views) . '</td>
							<td data-value="' . $comments . '">' . number_format($comments) . '</td>';

						$content .= '</tr>';

						$author_stats_data_additions = array(
							$author_id,
							$author_object->user_login,
							$author_object->display_name,
							$post['post'],
							$post['published_date'],
							$post['post_type'],
							$post['title'],
							$post['word_count'],
							$post_views,
							$comments,
						);

						$author_stats_data[] = $author_stats_data_additions;

					}

				$content .= '</tbody>
			</table>
			</div>
			<em class="text-muted" style="font-size: 12px;">* ' . __('Views and Comments are the only values affected by date range.', self::$text_domain) . '</em>';

			$comments_total = get_comments(array(
				'post_author' 	=> $author_id,
				'count' 		=> true,
				'date_query' 	=> array(
						array(
							'after'     => $start_date,
							'before'    => $end_date,
							'inclusive' => true,
					),
			)));

			$content = '<div class="panel panel-default ' . self::$prefix_dash . 'author-totals-container">' .
				'<div class="col-xs-12 panel-body">' .
					$author_data .
					'<div class="' . self::$prefix_dash . 'author-totals">' .
						'<div>' .
							'<p class="' . self::$prefix_dash . 'author-total-words">' . __('Total Posts', self::$text_domain) . '</p>' .
							'<div class="text-center">' . number_format( count($posts_by_author) ) . '</div>' .
						'</div>' .
						'<div>' .
							'<p class="' . self::$prefix_dash . 'author-total-words">' . __('Total Words', self::$text_domain) . '</p>' .
							'<div class="text-center">' . number_format( $total_word_count ) . '</div>' .
						'</div>' .
						'<div>' .
							'<p class="' . self::$prefix_dash . 'author-total-word-density">' . __('Average Word Count', self::$text_domain) . '</p>' .
							'<div class="text-center">' . number_format( (count($posts_by_author) != 0 ? round($total_word_count / count($posts_by_author), 2) : 0)) . '</div>' .
						'</div>' .
						'<div>' .
							'<p class="' . self::$prefix_dash . 'author-total-words">' . __('Total Views', self::$text_domain) . '</p>' .
							'<div class="text-center">' . number_format( NNRobots_Author_Stats_Table::get_views_by_author($author_id) ) . '</div>' .
						'</div>' .
						'<div>' .
							'<p class="' . self::$prefix_dash . 'author-total-comments">' . __('Total Comments', self::$text_domain) . '</p>' .
							'<div class="text-center">' . number_format( $comments_total ) . '</div>' .
						'</div>' .
					'</div>' .
				'</div>' .
			'</div>' . $content;

		}

		if ( !$export ) {
			echo json_encode(array('content' => $content));
			die();
		} else {
			return $author_stats_data;
		}

	}

	/**
	 * Hook for adding field to user profile page.
	 *
	 * @since 1.0.0
	 *
	 * @param 	N/A
	 * @return 	N/A
	 */
	static function add_field() {

		$settings = get_option(self::$prefix . 'settings');

		// Get settings

		if ( $settings == false ) {
			$settings = self::default_options();
		}

		$is_allowed = false;
		global $wp_roles;
		$current_user = wp_get_current_user();

		foreach ($wp_roles->role_names as $role_name => $role) {

			if (isset($settings['user_role_' .  $role_name]) && $settings['user_role_' .  $role_name] && in_array($role_name, $current_user->roles)) {
				$is_allowed = true;
			}
		}

		// Make sure to only display this if the current user can edit posts

		if ( current_user_can('edit_posts') && $is_allowed ) {

			// Style

			wp_enqueue_style(self::$prefix . 'settings_css', 					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/css/settings.css');
			wp_enqueue_style(self::$prefix . 'bootstrap_css', 					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/css/nnr-bootstrap.min.css');
			wp_enqueue_style(self::$prefix . 'fontawesome_css',					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/include/fontawesome/css/font-awesome.min.css');
			wp_enqueue_style(self::$prefix . 'bootstrap_sortable_css', 			NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/css/bootstrap-sortable.css');
			wp_enqueue_style(self::$prefix . 'bootstrap_selectize_css', 		NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/css/selectize.bootstrap3.css');
            wp_enqueue_style( 'jquery_ui_css', NNROBOTS_AUTHOR_STATS_PLUGIN_URL. '/admin/css/jquery-ui.css' );

            // Scripts
            wp_enqueue_script( 'jquery-ui-datepicker' );
			wp_enqueue_script(self::$prefix . 'bootstrap_js', 					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/bootstrap.min.js', array('jquery'));
			wp_enqueue_script(self::$prefix . 'bootstrap_sortable_js', 			NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/bootstrap-sortable.js', array('jquery'));
			wp_enqueue_script(self::$prefix . 'bootstrap_moment_js', 			NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/moment.js', array('jquery'));
			wp_enqueue_script(self::$prefix . 'bootstrap_selectize_js', 		NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/selectize.min.js', array('jquery'));
			wp_enqueue_script(self::$prefix . 'stats_user_js', 					NNROBOTS_AUTHOR_STATS_PLUGIN_URL . '/admin/js/stats-user.js', array('jquery'));
			wp_localize_script(self::$prefix . 'stats_user_js', 				'nnr_as_stats_user_data', array(
				'prefix_dash'			=> self::$prefix_dash,
				'today'					=> date('m/d/Y'),
				'post_types'			=> get_post_types(array('public' => true), 'names'),
			));
		}
	}

	/**
	 * get_user_profile_stats function.
	 *
	 * @access public
	 * @static
	 * @return void
	 */
	static function get_user_profile_stats() {

		$author_id = wp_get_current_user();

		// Format dates

		$start_date = isset($_POST["start_date"]) && $_POST["start_date"] != '' ?
						date('m/d/Y', strtotime($_POST["start_date"])) :
						date('m/d/Y', mktime(0, 0, 0, date("m"), date("d"), date("Y")-1));

		$end_date = isset($_POST["end_date"]) && $_POST["end_date"] != '' ?
						date('m/d/Y', strtotime($_POST["end_date"])) :
						date("m/d/Y", strtotime(current_time('mysql')));

		// Post Types

		if ( isset($_POST['post_types']) && $_POST['post_types'] != '' ) {
			$post_types = explode(',', urldecode($_POST['post_types']));
		} else {
			$post_types = array('post');// get_post_types(array('public' => true), 'names');
		}

        $data_filter_metric = !empty($_POST['filter_metric']) ? urldecode($_POST['filter_metric']) : 'posts';

        $settings = get_option(self::$prefix . 'settings');

		// Get settings

		if ( $settings == false ) {
			$settings = self::default_options();
		}

		$author_id = get_current_user_id();
		$author_object = get_user_by('id', $author_id);

		$posts_by_author = get_posts(array(
			'posts_per_page'	=> -1,
			'author'			=> $author_id,
			'post_type'			=> $post_types,
		));

		// Avatar

		$avatar = get_avatar($author_id, 50);

		$author_data =
			'<div class="' . self::$prefix_dash . 'author-avatar ' . self::$prefix_dash . 'single-author">' . $avatar . '</div>' .
			'<div class="' . self::$prefix_dash . 'author-data ' . self::$prefix_dash . 'single-author">' .
				'<div>' . $author_object->display_name . '</div>' .
				'<em class="' . self::$prefix_dash . 'author-id">id:(' . $author_id . ')</em>' .
			'</div>';

		$content_table = '<div class="clearfix"></div>
		<div class="' . self::$prefix_dash . 'table-container">
		<table class="table table-striped table-responsive sortable">
			<thead>
				<tr>
					<th>' . __('ID',self::$text_domain) . '</th>
					<th>' . __('Published Date',self::$text_domain) . '</th>
					<th>' . __('Post Type',self::$text_domain) . '</th>
					<th>' . __('Post Title',self::$text_domain) . '</th>
					<th>' . __('Word Count',self::$text_domain) . '</th>
					<th data-defaultsort="desc">' . __('Views *',self::$text_domain) . '</th>
					<th>' . __('Comments *',self::$text_domain) . '</th>';

				$content_table .= '</tr>
			</thead>

			<tbody>';

				$total_word_count = 0;

				foreach ($posts_by_author as $post ) {

					// Get word count

					$word_count = str_word_count( strip_tags($post->post_content) );
					$total_word_count += $word_count;

				    // Comments

				    $comments = get_comments(array(
						'post_id' 		=> $post->ID,
						'count' 		=> true,
						'date_query' 	=> array(
								array(
									'after'     => $start_date,
									'before'    => $end_date,
									'inclusive' => true,
							),
					)));

					$post_views = NNRobots_Author_Stats_Table::get_views_by_post($post->ID);

					// Content

					$content_table .= '<tr>
						<td>' . $post->ID . '</td>
						<td data-value="' . strtotime($post->post_date) . '">' . date('F j, Y', strtotime($post->post_date)) . '</td>
						<td>' . $post->post_type . '</td>
						<td><a href="' . get_the_permalink($post->ID) . '" target="_blank">' . wp_trim_words($post->post_title, 10, '...') . '</a></td>
						<td data-value="' . $word_count . '">' . number_format($word_count) . '</td>
						<td data-value="' . $post_views . '">' . number_format($post_views) . '</td>
						<td data-value="' . $comments . '">' . number_format($comments) . '</td>';

					$content_table .= '</tr>';

				}

			$content_table .= '</tbody>
		</table>
		</div>';

		$comments_total = get_comments(array(
			'post_author' 	=> $author_id,
			'count' 		=> true,
			'date_query' 	=> array(
					array(
						'after'     => $start_date,
						'before'    => $end_date,
						'inclusive' => true,
				),
		)));

		$content = '<div class="' . self::$prefix_dash . 'stats-wrap">
		<div class="nnr-wrap">
			<div class="nnr-container">

				<div class="' . self::$prefix_dash . 'form-filters">

					<button type="button" class="btn btn-default pull-left" data-toggle="modal" data-target="#' . self::$prefix_dash . 'advanced-settings"><i class="fa fa-cogs"></i> ' . __("Filters", self::$text_domain) . '</button>

					<div class="modal fade" style="z-index: 99999;" id="' . self::$prefix_dash . 'advanced-settings" tabindex="-1" role="dialog" aria-labelledby="myModalLabel">
						<div class="modal-dialog" role="document">
							<div class="modal-content">
								<div class="modal-header">
									<button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
									<h4 class="modal-title" id="myModalLabel">' . __("Advanced Settings", self::$text_domain) . '</h4>
								</div>
								<div class="modal-body" style="min-height: 300px;">

									<!-- Start Date -->

									<label class="control-label">' . __("Start Date", self::$text_domain) . '</label>

									<div class="input-group date ' . self::$prefix_dash . 'export-start-datepicker" id="' . self::$prefix_dash . 'start-datepicker">
										<input id="' . self::$prefix_dash . 'start-date" name="' . self::$prefix_dash . 'start-date" type="text" class="form-control" value="' . $start_date . '"/>
										<span class="input-group-addon"><span class="glyphicon glyphicon-calendar"></span>
										</span>
									</div>


									<em class="help-block">' . __("Select the start date to filter views and comments. Note: This will not filter posts by publish date", self::$text_domain) . '</em>

									<!-- End Date -->

									<label class="control-label">' . __("End Date", self::$text_domain) . '</label>

									<div class="input-group date ' . self::$prefix_dash . 'export-end-datepicker" id="' . self::$prefix_dash . 'end-datepicker">
										<input id="' . self::$prefix_dash . 'end-date" name="' . self::$prefix_dash . 'end-date" type="text" class="form-control" value="' . $end_date . '"/>
										<span class="input-group-addon"><span class="glyphicon glyphicon-calendar"></span>
										</span>
									</div>

									<em class="help-block">' . __("Select the start date to filter views and comments. Note: This will not filter posts by publish date", self::$text_domain) . '</em>


									<!-- Post Types -->

									<label for="' . self::$prefix_dash . 'post-types" class="control-label">' . __("Post Types", self::$text_domain) . '</label>
									<div class="form-group">
										<input style="width: 100%;" type="text" id="' . self::$prefix_dash . 'post-types" name="' . self::$prefix_dash . 'post-types" class="form-control" value="' . implode(',', $post_types) . '">
										<em class="help-block">' . __("Which post types should be in included in the stats data.", self::$text_domain) . '</em>
									</div>

								</div>
								<div class="modal-footer">
									<button class="btn btn-default ' . self::$prefix_dash . 'all-time" data-dismiss="modal"><i class="fa fa-clock-o"></i> ' . __("All Time", self::$text_domain) . '</button>
									<button class="btn btn-info ' . self::$prefix_dash . 'set-date" data-dismiss="modal"><i class="fa fa-filter"></i> ' . __("Apply", self::$text_domain) . '</button>
									<button class="btn btn-default pull-left" data-dismiss="modal">' . __("Cancel", self::$text_domain) . '</button>
								</div>
							</div>
						</div>
					</div>

				</div>

				<div class="clearfix"></div>

				<div class="' . self::$prefix_dash . 'stats-container">
					<div class="panel panel-default ' . self::$prefix_dash . 'author-totals-container">
						<div class="col-xs-12 panel-body">' .
							$author_data .
							'<div class="' . self::$prefix_dash . 'author-totals">' .
								'<div>' .
									'<p class="' . self::$prefix_dash . 'author-total-words">' . __('Total Published Posts', self::$text_domain) . '</p>' .
									'<div class="text-center">' . number_format(count($posts_by_author)) . '</div>' .
								'</div>' .
								'<div>' .
									'<p class="' . self::$prefix_dash . 'author-total-word-density">' . __('Average Word Count', self::$text_domain) . '</p>' .
									'<div class="text-center">' . number_format( (count($posts_by_author) != 0 ? round($total_word_count / count($posts_by_author), 2) : 0) ) . '</div>' .
								'</div>' .
								'<div>' .
									'<p class="' . self::$prefix_dash . 'author-total-words">' . __('Total Words', self::$text_domain) . '</p>' .
									'<div class="text-center">' . number_format($total_word_count) . '</div>' .
								'</div>' .
								'<div>' .
									'<p class="' . self::$prefix_dash . 'author-total-words">' . __('Total Views', self::$text_domain) . '</p>' .
									'<div class="text-center">' . number_format(NNRobots_Author_Stats_Table::get_views_by_author($author_id)) . '</div>' .
								'</div>' .
								'<div>' .
									'<p class="' . self::$prefix_dash . 'author-total-comments">' . __('Total Comments', self::$text_domain) . '</p>' .
									'<div class="text-center">' . number_format($comments_total) . '</div>' .
								'</div>' .
							'</div>
						</div>
					</div>
				</div>' . $content_table .
				'<em class="text-muted" style="font-size: 12px;">* ' . __('Views and Comments are the only values affected by date range.', self::$text_domain) . '</em>
			</div>
		</div>
		</div>';

		echo $content;

		die();
	}

	/**
	 * Export data to CSV
	 *
	 * @access public
	 * @static
	 * @param mixed $data
	 * @return void
	 */
	static function export_to_csv() {

		// Export data

		if (isset($_GET['export_data']) && $_GET['export_data'] == 'true' && check_admin_referer(self::$prefix . 'export_data')) {

			$filename 		= self::$prefix . 'all_authors.csv';
			$export_file 	= fopen(NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . 'exports/' . $filename, "w");

			if ( isset($_GET['author']) && $_GET['author'] != '' ) {
				$header = array('User ID', 'Username', 'Display Name', 'Post ID', 'Post Date', 'Post Type', 'Post Title', 'Word Count', 'Views', 'Comments');

			} else {
				$header = array('User ID', 'Username', 'Display Name', 'Posts', 'Average Word Count', 'Word Count', 'Views', 'Comments');

			}

			fputcsv($export_file, $header);

			$stats_data = self::stats_load(true);

			foreach ($stats_data as $stat_data) {
			    fputcsv($export_file, $stat_data);
			}

			fclose($export_file);

			@session_write_close();
			nocache_headers();
		    header('Content-Description: File Transfer');
		    header('Content-Type: application/octet-stream');
		    header('Content-Disposition: attachment; filename=' . basename(NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . 'exports/' . $filename));
		    header('Expires: 0');
		    header('Cache-Control: must-revalidate');
		    header('Pragma: public');
		    header('Content-Length: ' . filesize(NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . 'exports/' . $filename));
		    readfile(NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . 'exports/' . $filename);
		    exit;

		}

	}

	/**
	 * Update the word count for a post
	 *
	 * @access public
	 * @static
	 * @param mixed $post_id
	 * @param mixed $post
	 * @return void
	 */
	static function update_word_count($post_id, $post, $update) {

		if ( $post->post_type == 'revision' ) {
			return false;
		}

		include_once( NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . 'admin/model/indexed-posts.php');

		if ( $update ) {
			NNR_AS_Indexed_Post::update_post($post);
		} else {
			NNR_AS_Indexed_Post::add_post($post);
		}

	}

	/**
	 * Update the post status if the post is changed from published to anything else
	 *
	 * @access public
	 * @static
	 * @param mixed $new_status
	 * @param mixed $old_status
	 * @param mixed $post
	 * @return void
	 */
	static function update_post_status( $new_status, $old_status, $post ) {

		include_once( NNROBOTS_AUTHOR_STATS_PLUGIN_DIR . 'admin/model/indexed-posts.php');

	    if ( $old_status == 'publish'  &&  $new_status != 'publish' ) {
	        NNR_AS_Indexed_Post::delete_post($post->ID);
	    }

	    if ( $old_status != 'publish'  &&  $new_status == 'publish' ) {
			NNR_AS_Indexed_Post::add_post($post);
	    }
	}
}
