<?php

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

add_action('wp_footer', array('NNRobots_Author_Stats_Table', 'add_view'));

/**
 * NNRobots_Author_Stats_Table class.
 */
class NNRobots_Author_Stats_Table {

	/**
	 * table_name
	 *
	 * (default value: 'nnrobots_author_stats')
	 *
	 * @var string
	 * @access private
	 */
	private static $table_name = 'nnrobots_author_stats';

	/**
	 * Creates the DB table
	 *
	 * @access public
	 * @static
	 * @return void
	 */
	static function createTable() {

		global $wpdb;

		$wpdb->query(
			" CREATE TABLE IF NOT EXISTS " . self::getTableName() . " (
				`id` INT NOT NULL AUTO_INCREMENT PRIMARY KEY,
				`author` INT(11) NOT NULL,
				`post` INT(11) NOT NULL,
				`post_type` VARCHAR(60) NOT NULL DEFAULT 'post',
				`views` INT(11) NOT NULL,
				`date` date NOT NULL
			) ENGINE = InnoDB;");
	}

	/**
	 * Adds a view
	 *
	 * @access public
	 * @static
	 * @param mixed $author
	 * @param mixed $date
	 * @return void
	 */
	static function add_view() {

		// Return if this is not a single post

		if ( !is_singular() ) {
			return;
		}

		global $post;
		global $wpdb;

		// Check if the post is set

		if ( !isset($post->ID) ) {
			return;
		}

		// Get post author

		$author = $post->post_author;
		$date = date(NNROBOTS_AUTHOR_STATS_DATE_FORMAT);

		// Do not count view if user is the author or an editor or higher

		if ( $author == get_current_user_id() || current_user_can('manage_categories') ) {
			return;
		}

		// Check if entry already exsits

		$author_views = $wpdb->query($wpdb->prepare('SELECT * FROM ' . self::getTableName() . ' WHERE `date` = %s AND `author` = %d AND `post` = %d', $date, $author, $post->ID));

		// Entry aleady exists, just add 1

		if ( isset($author_views) && $author_views != 0 ) {

			$query = $wpdb->prepare('UPDATE ' . self::getTableName() . ' SET
				`views` = `views` + 1
				WHERE `date` = %s AND `author` = %d AND `post` = %d', $date, $author, $post->ID);

			$result = $wpdb->query($query);

		}

		// Entry does not exist, create a new one and set impressions to 1

		else {
			$query = $wpdb->prepare('INSERT INTO ' . self::getTableName() . ' (
				`author`,
				`post`,
				`post_type`,
				`views`,
				`date`
				) VALUES (%d, %d, %s, 1, %s)',
				$author,
				$post->ID,
				$post->post_type,
				$date
			);

			$result = $wpdb->query($query);
		}
	}

	/**
	 * Gets all the views for a specific author
	 *
	 * @access public
	 * @static
	 * @param mixed $author
	 * @return void
	 */
	static function get_views($start = null, $end = null, $author = null, $select = '*') {

		self::createTable();

		$query = null;

		global $wpdb;

		// All Authors, All Time

		if ($start == null && $end == null && $author == null) {
			$query = 'SELECT ' . $select . ' FROM ' . self::getTableName();
		}

		// All Views, After Date

		else if ($start != null && $end == null && $author == null) {
			$query = $wpdb->prepare('SELECT ' . $select . ' FROM ' . self::getTableName() . ' WHERE `date` = %s', $start);
		}

		// All Views, Date Range

		else if ($start != null && $end != null && $author == null) {
			$query = $wpdb->prepare('SELECT ' . $select . ' FROM ' . self::getTableName() . ' WHERE `date` >= %s AND `date` <= %s', $start, $end);
		}

		// Single Author, All Time

		else if ($start == null && $end == null && $author != null) {
			$query = $wpdb->prepare('SELECT ' . $select . ' FROM ' . self::getTableName() . ' WHERE `author` = %d', $author);
		}

		// Single Author, Date Range

		else if ($start != null && $end != null && $author != null) {
			$query = $wpdb->prepare('SELECT ' . $select . ' FROM ' . self::getTableName() . ' WHERE `date` >= %s AND `date` <= %s AND `author` = %d', $start, $end, $author);
		}

		// No query was created

		if (!isset($query)) {
			return false;
		}

		$result = $wpdb->get_results($query, 'ARRAY_A');

		return $result;
	}

	/**
	 * Gets all the views for a specific author
	 *
	 * @access public
	 * @static
	 * @param mixed $author
	 * @return void
	 */
	static function get_views_by_author($author, $post_types = array()) {

		// Get all post types if none are set

		if ( !is_array($post_types) || count($post_types) < 1 ) {
			$post_types = get_post_types('', 'names');
		}

		// If author is not set then return nothing

		if ( !isset($author) ) {
			return null;
		}

		global $wpdb;

		$results = $wpdb->get_results($wpdb->prepare('SELECT * FROM ' . self::getTableName() . ' WHERE `author` = %d', $author), 'ARRAY_A');

		// No results then return nothing

		if ( !isset($results) ) {
			return null;
		}

		$total_views = 0;

		foreach ( $results as $result ) {

			if ( in_array($result['post_type'], $post_types) ) {
				$total_views += $result['views'];
			}
		}

		return $total_views;
	}

	/**
	 * Gets all the views for a specific post
	 *
	 * @access public
	 * @static
	 * @param mixed $author
	 * @return void
	 */
	static function get_views_by_post($post, $start = null, $end = null) {

		// If post is not set then return nothing

		if ( !isset($post) ) {
			return null;
		}

		global $wpdb;

		$query = null;

		// All Views

		if ( $start == null && $end == null ) {
			$query = $wpdb->prepare('SELECT * FROM ' . self::getTableName() . ' WHERE `post` = %d', $post);
		}

		// All Views, After Date

		else if ( $start != null && $end == null ) {
			$query = $wpdb->prepare('SELECT ' . $select . ' FROM ' . self::getTableName() . ' WHERE `post` = %d AND `date` >= %s',$post, $start);
		}

		// All Views, Date Range

		else if ( $start != null && $end != null ) {
			$query = $wpdb->prepare('SELECT * FROM ' . self::getTableName() . ' WHERE `post` = %d AND `date` >= %s AND `date` <= %s', $post, $start, $end);
		}

		$results = $wpdb->get_results($query, 'ARRAY_A');

		// No results then return nothing

		if ( !isset($results) ) {
			return null;
		}

		$total_views = 0;

		foreach ( $results as $result ) {
			$total_views += $result['views'];
		}

		return $total_views;
	}

	/**
	 * Returns the right table name
	 *
	 * @access public
	 * @static
	 * @return void
	 */
	static function getTableName() {

		global $wpdb;

	    return '`' . $wpdb->prefix . self::$table_name . '`';
	}
}
