<?php

// Exit if accessed directly

if ( !defined( 'ABSPATH' ) ) exit;

/**
 * NNR_AS_Indexed_Post class.
 */
class NNR_AS_Indexed_Post {

	/**
	 * table_name
	 *
	 * (default value: 'nnr_as_indexed_posts')
	 *
	 * @var string
	 * @access private
	 */
	private static $table_name = 'nnr_as_indexed_posts';

	/**
	 * date_format
	 *
	 * (default value: 'Y-m-d H:i:s')
	 *
	 * @var string
	 * @access public
	 */
	static public $date_format = 'Y-m-d H:i:s';

	/**
	 * Creates the DB table
	 *
	 * @access public
	 * @static
	 * @return void
	 */
	static function createTable() {

		global $wpdb;

		$wpdb->query(
			" CREATE TABLE IF NOT EXISTS " . self::getTableName() . " (
				`id` INT NOT NULL AUTO_INCREMENT PRIMARY KEY,
				`post` INT(20) NOT NULL UNIQUE,
				`post_author` INT(20) NOT NULL,
				`published_date` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',
				`post_type` text NOT NULL DEFAULT '',
				`title` VARCHAR(60) NOT NULL DEFAULT 'post',
				`word_count` INT(20) NOT NULL
			) ENGINE = InnoDB;");
	}

	/**
	 * Adds a post
	 *
	 * @access public
	 * @static
	 * @param mixed $post
	 * @return void
	 */
	static function add_post($post) {

		global $wpdb;

		// if a post status is draft, ignore adding it
		if(in_array($post->post_status, array('draft', 'auto-draft'))) {
		    return;
        }

		$content   = preg_replace("/&#?[a-z0-9]{2,8};/i","", $post->post_content);
		$content   = strip_tags($content);
		$content   = preg_replace("~(?:\[/?)[^/\]]+/?\]~s", '', $content);

		$result = $wpdb->query( $wpdb->prepare("INSERT IGNORE INTO " . self::getTableName() . " (
				`post`,
				`post_author`,
				`published_date`,
				`post_type`,
				`title`,
				`word_count`
			) VALUES (%d, %d, %s, %s, %s, %d)
			ON DUPLICATE KEY update `post` = %d, `post_author` = %d, `published_date` = %s, `post_type` = %s, 
			`title` = %s, `word_count` = %d
			",
				$post->ID,
				$post->post_author,
				date(self::$date_format, strtotime($post->post_date)),
				$post->post_type,
				$post->post_title,
				str_word_count($content),
                $post->ID,
                $post->post_author,
                date(self::$date_format, strtotime($post->post_date)),
                $post->post_type,
                $post->post_title,
                str_word_count($content)
		) );

		// Return the recently created id for this entry

		return $wpdb->insert_id;
	}

	/**
	 * Update the post
	 *
	 * @access public
	 * @static
	 * @param mixed $post
	 * @param mixed $word_count
	 * @return void
	 */
	static function update_post($post) {

		global $wpdb;

		$content   = preg_replace("/&#?[a-z0-9]{2,8};/i","", $post->post_content);
		$content   = strip_tags($content);
		$content   = preg_replace("~(?:\[/?)[^/\]]+/?\]~s", '', $content);

		$result = $wpdb->query( $wpdb->prepare("UPDATE " . self::getTableName() . " SET
				`post_author` = %s,
				`published_date` = %s,
				`post_type` = %s,
				`title` = %s,
				`word_count` = %d
				WHERE `post` = %d",
				$post->post_author,
				date(self::$date_format, strtotime($post->post_date)),
				$post->post_type,
				$post->post_title,
				str_word_count($content),
				$post->ID
		) );

		// Return the recently created id for this entry

		return $wpdb->insert_id;

	}

	/**
	 * Gets all the posts for a specific author
	 *
	 * @access public
	 * @static
	 * @param mixed $author
	 * @return void
	 */
	static function get_posts($start = null, $end = null, $author = null, $select = '*') {

		$query = null;

		global $wpdb;

		// All Authors, All Time

		if ($start == null && $end == null && $author == null) {
			$query = 'SELECT ' . $select . ' FROM ' . self::getTableName();
		}

		// All Views, After Date

		else if ($start != null && $end == null && $author == null) {
			$query = $wpdb->prepare('SELECT ' . $select . ' FROM ' . self::getTableName() . ' WHERE `published_date` = %s', $start);
		}

		// All Views, Date Range

		else if ($start != null && $end != null && $author == null) {
			$query = $wpdb->prepare('SELECT ' . $select . ' FROM ' . self::getTableName() . ' WHERE `published_date` >= %s AND `published_date` <= %s', $start, $end);
		}

		// Single Author, All Time

		else if ($start == null && $end == null && $author != null) {
			$query = $wpdb->prepare('SELECT ' . $select . ' FROM ' . self::getTableName() . ' WHERE `post_author` = %d', $author);
		}

		// Single Author, Date Range

		else if ($start != null && $end != null && $author != null) {
			$query = $wpdb->prepare('SELECT ' . $select . ' FROM ' . self::getTableName() . ' WHERE `published_date` >= %s AND `published_date` <= %s AND `post_author` = %d', $start, $end, $author);
		}

		// No query was created

		if (!isset($query)) {
			return false;
		}

		$result = $wpdb->get_results($query, 'ARRAY_A');

		return $result;
	}

	/**
	 * Gets all the posts for a specific author
	 *
	 * @access public
	 * @static
	 * @param mixed $author
	 * @return void
	 */
	static function get_posts_by_author($author, $post_types = array(), $start_date = null, $end_date = null) {

		global $wpdb;

		// Get all post types if none are set

		if ( !is_array($post_types) || count($post_types) < 1 ) {
			$post_types = get_post_types('', 'names');
		}

		$post_types_list = '(';

		$count = 1;
		foreach ( $post_types as $post_type ) {

			if ( $count == count($post_types) ) {
				$post_types_list .= '"' . $post_type . '"';
			} else {
				$post_types_list .= '"' . $post_type . '", ';
			}

			$count++;
		}

		$post_types_list .= ')';

		// Start and End Dates

		$date_range = '';

		if ( isset( $start_date ) ) {
				$date_range .= $wpdb->prepare(' AND `published_date` >= %s', $start_date);
		}

		if ( isset( $end_date ) ) {
				$date_range .= $wpdb->prepare(' AND `published_date` <= %s', $end_date);
		}

		// If author is not set then return nothing

		if ( !isset($author) ) {
			return null;
		}

		$results = $wpdb->get_results($wpdb->prepare('SELECT * FROM ' . self::getTableName() . ' WHERE `post_author` = %d AND `post_type` IN ' . $post_types_list . ' ' . $date_range, $author), 'ARRAY_A');

		// No results then return nothing

		if ( !isset($results) ) {
			return null;
		}

		return $results;
	}

	/**
	 * Index all the posts for a site
	 *
	 * @access public
	 * @static
	 * @return void
	 */
	static function index_posts($page) {

		$posts = new WP_Query( array(
			'posts_per_page' 	=> 250,
			'paged'				=> $page,
			'post_type'			=> 'any',
            'post_status'       => 'publish'
		) );

		foreach ( $posts->posts as $post ) {
			self::add_post($post);
		}

		$posts = new WP_Query( array(
			'posts_per_page' 	=> 250,
			'paged'				=> $page + 1,
			'post_type'			=> 'any',
            'post_status'       => 'publish'
		) );

		if ( count($posts->posts) > 0 ) {
			$page++;
			wp_schedule_single_event( time() , 'nnr_as_index_all_posts', array($page) );
		} else {
			delete_transient('nnr_as_indexed_posts');

			$nnr_subject = "Author Stats Pro: Indexing Posts Successfully Completed - " . get_bloginfo('name');
			$nnr_body = "Good news! Author Stats Pro has successfully completed indexing and generating stats on ".get_site_url().". <br><br>Click the link below to view your site’s Author Stats: <br>" . get_admin_url() . 'users.php?page=' . NNRobots_Author_Stats::$settings_page;
			$nnr_body .= "<br><br>Thanks for using Author Stats Pro by the <a href='https://draftpress.com' target='_blank'>DraftPress</a> / 99 Robots team";
            $nnr_mail_headers = array('Content-Type: text/html; charset=UTF-8');
			wp_mail(
				get_option('admin_email'),
				__($nnr_subject, NNRobots_Author_Stats::$text_domain),
				__($nnr_body, NNRobots_Author_Stats::$text_domain),
                $nnr_mail_headers
			);
		}
	}

	/**
	 * Delete a post
	 *
	 * @access public
	 * @static
	 * @param mixed $post_id
	 * @return void
	 */
	static function delete_post($post_id) {

		if ( !isset($post_id) ) {
			return false;
		}

		global $wpdb;

		$results = $wpdb->get_results($wpdb->prepare('DELETE FROM ' . self::getTableName() . ' WHERE `post` = %d', $post_id), 'ARRAY_A');

		return $results;
	}

	/**
	 * Returns the right table name
	 *
	 * @access public
	 * @static
	 * @return void
	 */
	static function getTableName() {

		global $wpdb;

	    return '`' . $wpdb->prefix . self::$table_name . '`';
	}
}
